/**********************************************************************
*	Bastion Products Copyright 1998								*
*	All rights reserved										*
*	This source is freely distributable in an unmodified state.		*
*	Project:			Blithe Shared Library						*
*	Source:			BlitheSetWindow.cpp						*
*	Last Modified:	5/23/98									*
*	Author:			Jennifer Weston							*
*	Description:		This project creates a shared library which	*
*					allows applications to have dynamic text in	*
*					different languages.						*
***********************************************************************/
#include "BlitheSetWindow.h"
#include "VuBackdrop.h"
#include "BlitheDictionary.h"
#include "BlitheLib.h"
#include <TextView.h>
#include <MenuField.h>
#include <CheckBox.h>
#include <Path.h>
#include <FindDirectory.h>
#include <File.h>
#include <Resources.h>
#include <cstdlib>
#include <Application.h>

// Used internally
bool get_notify(void);

/**********************************************************************
*	Method:		BlitheSetWindow (constructor)					*
*	Arguments:	BRect			frame						*
*				const char*		title						*
*				BlitheDictionary*	appDictionary					*
*				window_type		type							*
*				uint32			flags						*
*				uint32			workspaces					*
*	Returns:		<none>										*
*	Description:	Creates the Blithe settings window. Sets up all of	*
*				the views.									*
**********************************************************************/
BlitheSetWindow::BlitheSetWindow(BlitheDictionary *appDictionary) : BWindow(BRect(150,100,500,250),"temp",B_TITLED_WINDOW,B_NOT_RESIZABLE|B_NOT_ZOOMABLE,B_CURRENT_WORKSPACE)
{
// Put in a background
	VuBackdrop *theView = new VuBackdrop(this);

	float rightBounds = Bounds().right-5;
// Put up a text field with instructions
	BRect bounds = theView->Bounds();
	bounds.InsetBy(5,5);
	bounds.bottom = 85;
	BRect interior = bounds;
	interior.OffsetTo(0,0);
	BTextView *theTextView = new BTextView(bounds,"instructions",interior,B_FOLLOW_LEFT | B_FOLLOW_TOP,B_WILL_DRAW);
	theTextView->MakeEditable(false);
	theTextView->MakeSelectable(false);
	theView->AddChild(theTextView);
	theTextView->SetViewColor(theView->ViewColor());

// Put in the popup menu
	BPopUpMenu *thePopup = appDictionary->LanguagePopupMenu();
	BMenuField *theMField = new BMenuField(BRect(10,90,200,105),"menubar","temp",thePopup);
	theView->AddChild(theMField);

// Put up a check box which, when on, notifies all running applications that the language has changed
	BCheckBox *theCheckBox = new BCheckBox(BRect(10,120,rightBounds,135),"notify","temp",NULL);
	theCheckBox->SetValue((get_notify()) ? B_CONTROL_ON : B_CONTROL_OFF);
	theView->AddChild(theCheckBox);

// Place the window
	BPath settingsPath;
	status_t err = find_directory(B_USER_SETTINGS_DIRECTORY,&settingsPath);
	if (err != B_NO_ERROR) return;
	settingsPath.Append("blithe_settings");
	BFile *settingsFile = new BFile(settingsPath.Path(),B_READ_ONLY);
	if (!settingsFile || settingsFile->InitCheck() != B_NO_ERROR) return;
	BResources *settingsRes = new BResources(settingsFile,false);
	delete settingsFile;
	if (settingsRes)
	{
		size_t possize;
		BPoint *pos = (BPoint*)(settingsRes->FindResource('winp',1,&possize));
		if (pos)
		{
			MoveTo(*pos);
			free(pos);
		}
	}
	delete settingsRes;
}

/**********************************************************************
*	Method:		~BlitheSetWindow (destructor)					*
*	Arguments:	<none>										*
*	Returns:		<none>										*
*	Description:	Deletes the window object; cleans up loose ends	*
**********************************************************************/
BlitheSetWindow::~BlitheSetWindow(void)
{
// Save off the notify state
	int32 id = get_default_language();
	bool notify = ShouldNotify();
	BRect frame = Frame();
	BPoint pos;
	pos.x = frame.left;
	pos.y = frame.top;
	BPath settingsPath;
	status_t err;
	err = find_directory(B_USER_SETTINGS_DIRECTORY,&settingsPath);
	if (err == B_NO_ERROR)
	{
		settingsPath.Append("blithe_settings");
		BFile settingsFile(settingsPath.Path(),B_READ_WRITE|B_CREATE_FILE|B_ERASE_FILE);
		BResources settingsRes(&settingsFile,true);
		err = settingsRes.AddResource('LONG',1,&id,sizeof(int32));
		err = settingsRes.AddResource('blnt',1,&notify,sizeof(bool));
		err = settingsRes.AddResource('winp',1,&pos,sizeof(pos));
	}

// Tell app to quit
	be_app->PostMessage(B_QUIT_REQUESTED);
}

/**********************************************************************
*	Method:		~ShouldNotify									*
*	Arguments:	<none>										*
*	Returns:		bool											*
*	Description:	Returns true if the notify check box is on		*
**********************************************************************/
bool BlitheSetWindow::ShouldNotify(void)
{
// Get the check box
	BCheckBox *notifyBox = dynamic_cast<BCheckBox*>(FindView("notify"));
	if (!notifyBox) return false;
	return (notifyBox->Value() == B_CONTROL_ON);
}